"""
Scalar Field Resonance Detector - Data Analysis Protocol
=========================================================
Supplementary Material for White Paper #1: SFT/STT

Author: Armando Zaragoza, USTE Technologies
Email: armando@uste-technologies.com
ORCID: 0009-0007-3542-0979
Date: November 25, 2025
License: CC-BY 4.0

NOTE: This is a PROPOSED experimental protocol. The experiment has not
yet been conducted. Use this script when experimental data is available.

Theoretical Prediction:
    - Resonance frequency: ~47 kHz
    - Signal amplification: ~1.8× relative to baseline
"""

import numpy as np
import pandas as pd
from scipy import stats
from scipy.signal import find_peaks

# Theoretical predictions
PREDICTED_FREQ_KHZ = 47.3
PREDICTED_AMPLIFICATION = 1.8

def load_data(filepath):
    """Load experimental CSV data."""
    return pd.read_csv(filepath)

def analyze_data(data):
    """
    Analyze resonance detector data.
    
    Expected CSV columns: freq_khz, trial_1, trial_2, ..., trial_10
    """
    trial_cols = [c for c in data.columns if c.startswith('trial_')]
    trials = data[trial_cols].values
    
    freq = data['freq_khz'].values
    mean_v = np.mean(trials, axis=1)
    std_v = np.std(trials, axis=1, ddof=1)
    baseline = np.median(mean_v)
    amplification = mean_v / baseline
    
    # Find peaks
    peaks, _ = find_peaks(mean_v, height=baseline*1.2, prominence=baseline*0.1)
    
    results = {
        'freq': freq,
        'mean': mean_v,
        'std': std_v,
        'baseline': baseline,
        'amplification': amplification,
        'peaks': peaks
    }
    
    if len(peaks) > 0:
        max_idx = peaks[np.argmax(mean_v[peaks])]
        results['peak_freq'] = freq[max_idx]
        results['peak_amp'] = amplification[max_idx]
        
        # Statistical significance
        z = (mean_v[max_idx] - baseline) / (std_v[max_idx] / np.sqrt(len(trial_cols)))
        results['z_score'] = z
        results['p_value'] = 2 * (1 - stats.norm.cdf(abs(z)))
    
    return results

def generate_test_data(include_resonance=True):
    """Generate simulated data for testing the analysis pipeline."""
    freq = np.arange(1, 101, 1)
    data = {'freq_khz': freq}
    
    for i in range(1, 11):
        v = np.ones_like(freq, dtype=float)
        if include_resonance:
            gamma = 3.0
            peak = 0.8 * (gamma**2) / ((freq - PREDICTED_FREQ_KHZ)**2 + gamma**2)
            v += peak
        v += np.random.normal(0, 0.05, len(freq))
        data[f'trial_{i}'] = v
    
    return pd.DataFrame(data)

def print_report(results):
    """Print analysis report."""
    print("=" * 60)
    print("RESONANCE DETECTOR ANALYSIS REPORT")
    print("=" * 60)
    
    print(f"\nBaseline voltage: {results['baseline']:.4f} V")
    print(f"Peaks detected: {len(results['peaks'])}")
    
    if 'peak_freq' in results:
        print(f"\nMaximum peak:")
        print(f"    Frequency: {results['peak_freq']:.1f} kHz")
        print(f"    Amplification: {results['peak_amp']:.2f}×")
        print(f"    Z-score: {results['z_score']:.2f}")
        print(f"    P-value: {results['p_value']:.2e}")
        
        sig = "SIGNIFICANT" if results['p_value'] < 0.05 else "NOT SIGNIFICANT"
        print(f"    Status: {sig}")
        
        print(f"\nComparison to SFT/STT predictions:")
        print(f"    Predicted: {PREDICTED_FREQ_KHZ} kHz, {PREDICTED_AMPLIFICATION}×")
        print(f"    Observed:  {results['peak_freq']:.1f} kHz, {results['peak_amp']:.2f}×")
        
        freq_dev = abs(results['peak_freq'] - PREDICTED_FREQ_KHZ) / PREDICTED_FREQ_KHZ * 100
        amp_dev = abs(results['peak_amp'] - PREDICTED_AMPLIFICATION) / PREDICTED_AMPLIFICATION * 100
        print(f"    Frequency deviation: {freq_dev:.1f}%")
        print(f"    Amplitude deviation: {amp_dev:.1f}%")
        
        if freq_dev < 10 and amp_dev < 20:
            print("\n    ✓ CONSISTENT WITH SFT/STT PREDICTIONS")
        else:
            print("\n    ✗ DEVIATES FROM PREDICTIONS")
    else:
        print("\nNo significant peaks detected.")
    
    print("=" * 60)

def main():
    print("Generating simulated test data...")
    data = generate_test_data(include_resonance=True)
    
    print("Analyzing data...")
    results = analyze_data(data)
    
    print_report(results)
    
    print("\nNOTE: This analysis used SIMULATED data for demonstration.")
    print("Replace with actual experimental data when available.")

if __name__ == "__main__":
    main()
